#!/bin/sh
set -e
set -u
set -o pipefail

function on_error {
  echo "$(realpath -mq "${0}"):$1: error: Unexpected failure"
}
trap 'on_error $LINENO' ERR

if [ -z ${UNLOCALIZED_RESOURCES_FOLDER_PATH+x} ]; then
  # If UNLOCALIZED_RESOURCES_FOLDER_PATH is not set, then there's nowhere for us to copy
  # resources to, so exit 0 (signalling the script phase was successful).
  exit 0
fi

mkdir -p "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"

RESOURCES_TO_COPY=${PODS_ROOT}/resources-to-copy-${TARGETNAME}.txt
> "$RESOURCES_TO_COPY"

XCASSET_FILES=()

# This protects against multiple targets copying the same framework dependency at the same time. The solution
# was originally proposed here: https://lists.samba.org/archive/rsync/2008-February/020158.html
RSYNC_PROTECT_TMP_FILES=(--filter "P .*.??????")

case "${TARGETED_DEVICE_FAMILY:-}" in
  1,2)
    TARGET_DEVICE_ARGS="--target-device ipad --target-device iphone"
    ;;
  1)
    TARGET_DEVICE_ARGS="--target-device iphone"
    ;;
  2)
    TARGET_DEVICE_ARGS="--target-device ipad"
    ;;
  3)
    TARGET_DEVICE_ARGS="--target-device tv"
    ;;
  4)
    TARGET_DEVICE_ARGS="--target-device watch"
    ;;
  *)
    TARGET_DEVICE_ARGS="--target-device mac"
    ;;
esac

install_resource()
{
  if [[ "$1" = /* ]] ; then
    RESOURCE_PATH="$1"
  else
    RESOURCE_PATH="${PODS_ROOT}/$1"
  fi
  if [[ ! -e "$RESOURCE_PATH" ]] ; then
    cat << EOM
error: Resource "$RESOURCE_PATH" not found. Run 'pod install' to update the copy resources script.
EOM
    exit 1
  fi
  case $RESOURCE_PATH in
    *.storyboard)
      echo "ibtool --reference-external-strings-file --errors --warnings --notices --minimum-deployment-target ${!DEPLOYMENT_TARGET_SETTING_NAME} --output-format human-readable-text --compile ${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename \"$RESOURCE_PATH\" .storyboard`.storyboardc $RESOURCE_PATH --sdk ${SDKROOT} ${TARGET_DEVICE_ARGS}" || true
      ibtool --reference-external-strings-file --errors --warnings --notices --minimum-deployment-target ${!DEPLOYMENT_TARGET_SETTING_NAME} --output-format human-readable-text --compile "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename \"$RESOURCE_PATH\" .storyboard`.storyboardc" "$RESOURCE_PATH" --sdk "${SDKROOT}" ${TARGET_DEVICE_ARGS}
      ;;
    *.xib)
      echo "ibtool --reference-external-strings-file --errors --warnings --notices --minimum-deployment-target ${!DEPLOYMENT_TARGET_SETTING_NAME} --output-format human-readable-text --compile ${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename \"$RESOURCE_PATH\" .xib`.nib $RESOURCE_PATH --sdk ${SDKROOT} ${TARGET_DEVICE_ARGS}" || true
      ibtool --reference-external-strings-file --errors --warnings --notices --minimum-deployment-target ${!DEPLOYMENT_TARGET_SETTING_NAME} --output-format human-readable-text --compile "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename \"$RESOURCE_PATH\" .xib`.nib" "$RESOURCE_PATH" --sdk "${SDKROOT}" ${TARGET_DEVICE_ARGS}
      ;;
    *.framework)
      echo "mkdir -p ${TARGET_BUILD_DIR}/${FRAMEWORKS_FOLDER_PATH}" || true
      mkdir -p "${TARGET_BUILD_DIR}/${FRAMEWORKS_FOLDER_PATH}"
      echo "rsync --delete -av "${RSYNC_PROTECT_TMP_FILES[@]}" $RESOURCE_PATH ${TARGET_BUILD_DIR}/${FRAMEWORKS_FOLDER_PATH}" || true
      rsync --delete -av "${RSYNC_PROTECT_TMP_FILES[@]}" "$RESOURCE_PATH" "${TARGET_BUILD_DIR}/${FRAMEWORKS_FOLDER_PATH}"
      ;;
    *.xcdatamodel)
      echo "xcrun momc \"$RESOURCE_PATH\" \"${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH"`.mom\"" || true
      xcrun momc "$RESOURCE_PATH" "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH" .xcdatamodel`.mom"
      ;;
    *.xcdatamodeld)
      echo "xcrun momc \"$RESOURCE_PATH\" \"${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH" .xcdatamodeld`.momd\"" || true
      xcrun momc "$RESOURCE_PATH" "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH" .xcdatamodeld`.momd"
      ;;
    *.xcmappingmodel)
      echo "xcrun mapc \"$RESOURCE_PATH\" \"${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH" .xcmappingmodel`.cdm\"" || true
      xcrun mapc "$RESOURCE_PATH" "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH" .xcmappingmodel`.cdm"
      ;;
    *.xcassets)
      ABSOLUTE_XCASSET_FILE="$RESOURCE_PATH"
      XCASSET_FILES+=("$ABSOLUTE_XCASSET_FILE")
      ;;
    *)
      echo "$RESOURCE_PATH" || true
      echo "$RESOURCE_PATH" >> "$RESOURCES_TO_COPY"
      ;;
  esac
}
if [[ "$CONFIGURATION" == "Debug" ]]; then
  install_resource "${PODS_ROOT}/ADAL/ADALiOS/ADALiOS/ADAL_iPad_Storyboard.storyboard"
  install_resource "${PODS_ROOT}/ADAL/ADALiOS/ADALiOS/ADAL_iPhone_Storyboard.storyboard"
  install_resource "${PODS_ROOT}/Extension/Classes/iOS/UIColor+PropertyDictionary/DefaultColor.plist"
  install_resource "${PODS_ROOT}/FBSDKCoreKit/FacebookSDKStrings.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseCore/FirebaseCore_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseCoreExtension/FirebaseCoreExtension_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseCoreInternal/FirebaseCoreInternal_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseCrashlytics/FirebaseCrashlytics_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseDynamicLinks/FirebaseDynamicLinks_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseInstallations/FirebaseInstallations_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/GoogleDataTransport/GoogleDataTransport_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/GoogleUtilities/GoogleUtilities_Privacy.bundle"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Image/PPAlbumContentViewControllerButtonAllSelect-Normal.png"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Image/PPAlbumContentViewControllerButtonImport-Highlighted.png"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Image/PPAlbumContentViewControllerButtonImport-Normal.png"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPBackgroundTaskController/Assets/PPBTC.wav"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/en.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/de.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/es.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/fr.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/it.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/ja.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/ko.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/nl.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/pt.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Classes/HDRGenerator/PPHDRGenerator.metallib"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageButtonChange-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageButtonMore-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageButtonRotateLeft-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageButtonRotateRight-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonAdd-Highlighted.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonAdd-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonAddRearCard-Highlighted.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonAddRearCard-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonBack-Highlighted.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonBack-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonCategory-Disabled.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonFlip-Highlighted.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonFlip-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPCheckBoxView/Assets/DefaultImageForCheckBoxViewSelected.png"
  install_resource "${PODS_ROOT}/PPCheckBoxView/Assets/DefaultImageForCheckBoxViewUnselected.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Image/PPClipBusinessCardControllerButtonChange-Highlighted.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Image/PPClipBusinessCardControllerButtonChange-Normal.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Image/PPClipBusinessCardControllerButtonFill-Normal.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Image/PPClipBusinessCardControllerButtonRecog-Highlighted.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Image/PPClipBusinessCardControllerButtonRecog-Normal.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/de.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/es.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/fr.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/it.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/ja.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/ko.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/nl.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/pt.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/en.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CN2CC.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_de.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_en.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_es.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_fr.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_it.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_ja.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_ko.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_nl.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_pt.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_zh-hans.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_zh-hant.plist"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPGuideController/Assets/PPGuideControllerFocusImage@2x.png"
  install_resource "${PODS_ROOT}/PPGuideController/Assets/PPGuideControllerIconImageSwipe@3x.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonClip-Highlighted.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonClip-Normal.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonFullRect-Highlighted.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonFullRect-Normal.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonLight-Highlighted.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonLight-Normal.png"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPMakeVirtualCardController/Assets/PPMakeVirtualCardControllerHeadImage.png"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/en.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/de.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/es.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/fr.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/it.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/ja.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/ko.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/nl.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/pt.lproj"
  install_resource "${PODS_ROOT}/PPSectionIndexView/Assets/PPSectionIndexView-Search-D.png"
  install_resource "${PODS_ROOT}/PPSectionIndexView/Assets/PPSectionIndexView-Search-N.png"
  install_resource "${PODS_ROOT}/PPSelectImageView/Assets/Gray/PPSelectImageViewSelect.png"
  install_resource "${PODS_ROOT}/PPSelectImageView/Assets/Gray/PPSelectImageViewSelectHighlighted.png"
  install_resource "${PODS_ROOT}/PPSelectImageView/Assets/Gray/PPSelectImageViewUnselect.png"
  install_resource "${PODS_ROOT}/PPSelectImageView/Assets/Gray/PPSelectImageViewUnselectHighlighted.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Image/PPStorageFileSelectViewControllerButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Image/PPStorageFileSelectViewControllerButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Image/PPStorageFileSelectViewControllerButtonEdit-Highlighted.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Image/PPStorageFileSelectViewControllerButtonEdit-Normal.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Image/PPStorageFileSelectViewControllerButtonImport-Normal.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPTableViewCell/PPButtonTableViewCell/Resources/Image/PPButtonTableViewCellSwitchOff.png"
  install_resource "${PODS_ROOT}/PPTableViewCell/PPButtonTableViewCell/Resources/Image/PPButtonTableViewCellSwitchOn.png"
  install_resource "${PODS_ROOT}/PPTableViewCell/PPMutipleTextFieldButtonTableViewCell/Resources/PPTextFieldButtonViewDetailDisclosureButton-Disabled.png"
  install_resource "${PODS_ROOT}/PPTableViewCell/PPMutipleTextFieldButtonTableViewCell/Resources/PPTextFieldButtonViewDetailDisclosureButton-Highlighted.png"
  install_resource "${PODS_ROOT}/PPTableViewCell/PPMutipleTextFieldButtonTableViewCell/Resources/PPTextFieldButtonViewDetailDisclosureButton-Normal.png"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/PromisesObjC/FBLPromises_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/PromisesSwift/Promises_Privacy.bundle"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Image/WCABPeoplePickerViewControllerButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Image/WCABPeoplePickerViewControllerButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Image/WCABPeoplePickerViewControllerButtonSelectAll-Highlighted.png"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Image/WCABPeoplePickerViewControllerButtonSelectAll-Normal.png"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCAlbumFlowController/Assets/Image/WCAlbumViewControllerButtonDownPull-Highlighted.png"
  install_resource "${PODS_ROOT}/WCAlbumFlowController/Assets/Image/WCAlbumViewControllerButtonDownPull-Normal.png"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseBottomViewButtonAgainCard-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseBottomViewButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseBottomViewButtonNextCard-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseBottomViewButtonRearCapture-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseTopViewButtonTurnLeft-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseTopViewButtonTurnLeft-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseTopViewButtonTurnRight-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseTopViewButtonTurnRight-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_AutoCrop-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_AutoCrop-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Cancel-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Cancel-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Capture-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Capture-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_CardDirectionLandscapeBackSide.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_CardDirectionLandscapeFrontSide.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_CardDirectionPortraitBackSide.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_CardDirectionPortraitFrontSide.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Done-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Done-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_DoubleSide-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_DoubleSide-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_FakeCardBackSide.jpg"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_FakeCardFrontSide.jpg"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Flash-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Flash-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Guide-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Guide-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_ImageEnhance-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_ImageEnhance-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Torch-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Torch-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_EMail-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_EMail-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_EMail.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Facetime-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Facetime-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Facetime.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Hangouts-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Hangouts-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Hangouts.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Map-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Map-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Map.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Skype-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Skype-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Skype.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_SMS-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_SMS-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_SMS.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Tel-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Tel-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Tel.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Checked.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Download-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Download-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Navi-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Navi-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Uncheck.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Unverified.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/NoteViewControllerButtonDate-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/NoteViewControllerButtonDate-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactControllerButtonGroupUnSynced.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactMoreButtonDelete-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactMoreButtonDelete-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactMoreButtonNote-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactMoreButtonNote-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonChangeSide-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonChangeSide-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonHideKeyboard-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonMore-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonMore-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonSave-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonSave-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditDefaultHead.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditMoreButtonChange-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditMoreButtonChange-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditMoreButtonClipAdd-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditMoreButtonClipAdd-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddFrontCard-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddFrontCard-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddNext-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddNext-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddRearCard-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddRearCard-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonFlip-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonFlip-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonNote-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonNote-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoDefaultHead-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonCommunitySearch-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonCommunitySearch-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonEdit-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonEdit-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonRecog-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonRecog-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonRefreshSocialNetwork-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonRefreshSocialNetwork-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonSaveContact-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonSaveContact-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonSendContact-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonSendContact-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonShareContact-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonShareContact-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonTranslate-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonTranslate-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonFavorite-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonFavorite-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonFavorite-Selected.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonLanguageArrow-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonLanguageArrow-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonMoreLanguage-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonMoreLanguage-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonNote-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonNote-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellAddress-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellBirthday-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellCompany-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellContactName-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellDate-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellEmail-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellGroup-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellIM-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellOthers-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellPhone-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellSMS-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellSMS-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellSocialNetwork-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellURL-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellUserDefine-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellViewAccount-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarAddress-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarAddress-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarAddress-Selected.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarGo-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarGo-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarMail-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarMail-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarMail-Selected.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarNext-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarNext-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarPrev-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarPrev-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarTel-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarTel-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarTel-Selected.png"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Image/WCExportFlowButtonExport-Highlighted.png"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Image/WCExportFlowButtonExport-Normal.png"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/ic_check-box_select_h.png"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/ic_check-box_unselect_h.png"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupAdd-Highlighted.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupAdd-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupDone-Highlighted.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupDone-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupSort-Highlighted.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupSort-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/de.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/en.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/es.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/fr.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/it.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/ja.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/ko.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/nl.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/pt.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/de.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/en.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/es.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/fr.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/it.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/ja.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/ko.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/nl.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/pt.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/Collapse-Highlighted.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/Collapse-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/Expand-Highlighted.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/Expand-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/Pin-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/UnPin-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/WCGroupTableViewCell_Checked.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/WCGroupTableViewCell_Uncheck.png"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Image/GroupViewControllerButtonImport-Highlighted.png"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Image/GroupViewControllerButtonImport-Normal.png"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerButtonLocation-Highlighted.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerButtonLocation-Normal.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerButtonOther-Normal.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerButtonSelf-Normal.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerDefaultHead.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerTableViewCellDisclosureIndicator-Highlighted.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerTableViewCellDisclosureIndicator-Normal.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/PPSignatureConrollerButtonRecog-Highlighted.png"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/PPSignatureConrollerButtonRecog-Normal.png"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/PPSignatureConrollerButtonTip-Highlighted.png"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/PPSignatureConrollerButtonTip-Normal.png"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/de.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/en.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/es.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/fr.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/it.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/ja.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/ko.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/nl.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/pt.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCSearchViewController/Assets/Images/WCSearchViewControllerButton_AdvanceSearch-Normal.png"
  install_resource "${PODS_ROOT}/WCSearchViewController/Assets/Images/WCSearchViewControllerButton_SearchClear-Highlighted@3x.png"
  install_resource "${PODS_ROOT}/WCSearchViewController/Assets/Images/WCSearchViewControllerButton_SearchClear-Normal@3x.png"
  install_resource "${PODS_ROOT}/WCSearchViewController/Assets/Images/WCSearchViewControllerButton_SearchIcon@3x.png"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Image/WCShareFlowButtonShare-Highlighted.png"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Image/WCShareFlowButtonShare-Normal.png"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoMoreButtonEdit-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoMoreButtonEdit-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoMoreButtonTemplate-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoMoreButtonTemplate-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoShareCellEmail-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoShareCellQRCode-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoShareCellSMS-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Image/AccountSelectDone-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Image/AccountSelectDone-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Image/ic_check-box_select_h.png"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Image/ic_check-box_unselect_h.png"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/VisibleAccount-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/VisibleAccount-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonChangeOwner-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonChangeOwner-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDelete-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDelete-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownload-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownload-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloaded-Disabled.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloaded-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloaded-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloading-Disabled.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloading-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloading-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonNeedDownload-Disabled.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonNeedDownload-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonNeedDownload-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonShareToColleague-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonShareToColleague-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCTCompanyMultiSelect-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCTCompanyMultiSelect-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCTCompanySearch-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCTCompanySearch-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonDrop-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonDrop-Normal.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonHidePassword-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonHidePassword-Normal.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonShowPassword-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonShowPassword-Normal.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerContentBackground.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerHeaderBackground.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTNoticePageViewController_Filter-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTNoticePageViewController_Filter-Normal.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTSharedContactNoticeViewController_Download-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTSharedContactNoticeViewController_Download-Normal.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTSharedContactNoticeViewController_SelectAll-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTSharedContactNoticeViewController_SelectAll-Normal.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Image/WCTPasswordViewControllerButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Image/WCTPasswordViewControllerButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Images/WCTELVC_ButtonHidePassword-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Images/WCTELVC_ButtonHidePassword-Normal.png"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Images/WCTELVC_ButtonShowPassword-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Images/WCTELVC_ButtonShowPassword-Normal.png"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Classes/Private/WCTQNapLoginViewController.xib"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Classes/Private/WCTQNapQuestionViewController.xib"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Classes/Private/WCTQNapSecurityCodeViewController.xib"
  install_resource "${PODS_ROOT}/WCXFDataController/Assets/wcxficon.png"
  install_resource "${PODS_ROOT}/WCXFDataController/Assets/wcxficon@2x.png"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/libIndexing/Assets/Default/ppkkc.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/amedict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/arabicdict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/bphrase1.big"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/bphrase1.gbk"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/bphrase1.jap"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/bphrase1.ksc"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/cocr_dts"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/cocr_sim"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/cocr_trad"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_ar.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_br.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_eeu.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_eng.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_gr.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_neu.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_proto_th.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_rus.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_th.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_tky.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_weu.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/engdict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/esteupdict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/eupdict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/gb-big5.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/grdict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/jocr"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/jocra"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/kocr"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/neudict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phb5.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phceng.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phgb.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phjeng.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phjp.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phks.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/russiadict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/surname.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/thaidict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/turkeydict.dat"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/nanopb/nanopb_Privacy.bundle"
fi
if [[ "$CONFIGURATION" == "Release" ]]; then
  install_resource "${PODS_ROOT}/ADAL/ADALiOS/ADALiOS/ADAL_iPad_Storyboard.storyboard"
  install_resource "${PODS_ROOT}/ADAL/ADALiOS/ADALiOS/ADAL_iPhone_Storyboard.storyboard"
  install_resource "${PODS_ROOT}/Extension/Classes/iOS/UIColor+PropertyDictionary/DefaultColor.plist"
  install_resource "${PODS_ROOT}/FBSDKCoreKit/FacebookSDKStrings.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseCore/FirebaseCore_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseCoreExtension/FirebaseCoreExtension_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseCoreInternal/FirebaseCoreInternal_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseCrashlytics/FirebaseCrashlytics_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseDynamicLinks/FirebaseDynamicLinks_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/FirebaseInstallations/FirebaseInstallations_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/GoogleDataTransport/GoogleDataTransport_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/GoogleUtilities/GoogleUtilities_Privacy.bundle"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Image/PPAlbumContentViewControllerButtonAllSelect-Normal.png"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Image/PPAlbumContentViewControllerButtonImport-Highlighted.png"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Image/PPAlbumContentViewControllerButtonImport-Normal.png"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPAlbumViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPBackgroundTaskController/Assets/PPBTC.wav"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/en.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/de.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/es.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/fr.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/it.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/ja.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/ko.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/nl.lproj"
  install_resource "${PODS_ROOT}/PPButton/Assets/Localizations/pt.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPCameraView/Classes/HDRGenerator/PPHDRGenerator.metallib"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageButtonChange-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageButtonMore-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageButtonRotateLeft-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageButtonRotateRight-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonAdd-Highlighted.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonAdd-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonAddRearCard-Highlighted.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonAddRearCard-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonBack-Highlighted.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonBack-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonCategory-Disabled.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonFlip-Highlighted.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Image/CardImageTopBarViewButtonFlip-Normal.png"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPCardImageViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPCheckBoxView/Assets/DefaultImageForCheckBoxViewSelected.png"
  install_resource "${PODS_ROOT}/PPCheckBoxView/Assets/DefaultImageForCheckBoxViewUnselected.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Image/PPClipBusinessCardControllerButtonChange-Highlighted.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Image/PPClipBusinessCardControllerButtonChange-Normal.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Image/PPClipBusinessCardControllerButtonFill-Normal.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Image/PPClipBusinessCardControllerButtonRecog-Highlighted.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Image/PPClipBusinessCardControllerButtonRecog-Normal.png"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/de.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/es.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/fr.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/it.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/ja.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/ko.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/nl.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/pt.lproj"
  install_resource "${PODS_ROOT}/PPClipBusinessCardViewController/Assets/Localizations/en.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPClipContactPhotoViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CN2CC.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_de.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_en.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_es.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_fr.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_it.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_ja.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_ko.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_nl.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_pt.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_zh-hans.plist"
  install_resource "${PODS_ROOT}/PPCountryCodeConvert/Assets/CountryTemplate_zh-hant.plist"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPDataConvertFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPGuideController/Assets/PPGuideControllerFocusImage@2x.png"
  install_resource "${PODS_ROOT}/PPGuideController/Assets/PPGuideControllerIconImageSwipe@3x.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonClip-Highlighted.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonClip-Normal.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonFullRect-Highlighted.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonFullRect-Normal.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonLight-Highlighted.png"
  install_resource "${PODS_ROOT}/PPImageEnhanceViewController/Assets/Image/PPImageEnhanceViewControllerButtonLight-Normal.png"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPLockScreenController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPMakeVirtualCardController/Assets/PPMakeVirtualCardControllerHeadImage.png"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/en.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/de.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/es.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/fr.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/it.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/ja.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/ko.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/nl.lproj"
  install_resource "${PODS_ROOT}/PPPromotionViewController/LocalizableResource/pt.lproj"
  install_resource "${PODS_ROOT}/PPSectionIndexView/Assets/PPSectionIndexView-Search-D.png"
  install_resource "${PODS_ROOT}/PPSectionIndexView/Assets/PPSectionIndexView-Search-N.png"
  install_resource "${PODS_ROOT}/PPSelectImageView/Assets/Gray/PPSelectImageViewSelect.png"
  install_resource "${PODS_ROOT}/PPSelectImageView/Assets/Gray/PPSelectImageViewSelectHighlighted.png"
  install_resource "${PODS_ROOT}/PPSelectImageView/Assets/Gray/PPSelectImageViewUnselect.png"
  install_resource "${PODS_ROOT}/PPSelectImageView/Assets/Gray/PPSelectImageViewUnselectHighlighted.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Image/PPStorageFileSelectViewControllerButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Image/PPStorageFileSelectViewControllerButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Image/PPStorageFileSelectViewControllerButtonEdit-Highlighted.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Image/PPStorageFileSelectViewControllerButtonEdit-Normal.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Image/PPStorageFileSelectViewControllerButtonImport-Normal.png"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPStorageFileSelectViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/PPStorageSpaceSelectViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/PPTableViewCell/PPButtonTableViewCell/Resources/Image/PPButtonTableViewCellSwitchOff.png"
  install_resource "${PODS_ROOT}/PPTableViewCell/PPButtonTableViewCell/Resources/Image/PPButtonTableViewCellSwitchOn.png"
  install_resource "${PODS_ROOT}/PPTableViewCell/PPMutipleTextFieldButtonTableViewCell/Resources/PPTextFieldButtonViewDetailDisclosureButton-Disabled.png"
  install_resource "${PODS_ROOT}/PPTableViewCell/PPMutipleTextFieldButtonTableViewCell/Resources/PPTextFieldButtonViewDetailDisclosureButton-Highlighted.png"
  install_resource "${PODS_ROOT}/PPTableViewCell/PPMutipleTextFieldButtonTableViewCell/Resources/PPTextFieldButtonViewDetailDisclosureButton-Normal.png"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/PromisesObjC/FBLPromises_Privacy.bundle"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/PromisesSwift/Promises_Privacy.bundle"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Image/WCABPeoplePickerViewControllerButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Image/WCABPeoplePickerViewControllerButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Image/WCABPeoplePickerViewControllerButtonSelectAll-Highlighted.png"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Image/WCABPeoplePickerViewControllerButtonSelectAll-Normal.png"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCABItemSelectFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCAdvancedSearchViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCAlbumFlowController/Assets/Image/WCAlbumViewControllerButtonDownPull-Highlighted.png"
  install_resource "${PODS_ROOT}/WCAlbumFlowController/Assets/Image/WCAlbumViewControllerButtonDownPull-Normal.png"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCBackupRestoreFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseBottomViewButtonAgainCard-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseBottomViewButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseBottomViewButtonNextCard-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseBottomViewButtonRearCapture-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseTopViewButtonTurnLeft-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseTopViewButtonTurnLeft-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseTopViewButtonTurnRight-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCaptureFlowController/Assets/WCImageBrowseViewController/WCImageBrowseTopViewButtonTurnRight-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_AutoCrop-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_AutoCrop-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Cancel-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Cancel-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Capture-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Capture-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_CardDirectionLandscapeBackSide.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_CardDirectionLandscapeFrontSide.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_CardDirectionPortraitBackSide.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_CardDirectionPortraitFrontSide.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Done-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Done-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_DoubleSide-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_DoubleSide-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_FakeCardBackSide.jpg"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_FakeCardFrontSide.jpg"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Flash-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Flash-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Guide-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Guide-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_ImageEnhance-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_ImageEnhance-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Torch-Normal.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Images/WCMCC_Torch-Selected.png"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCCaptureViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_EMail-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_EMail-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_EMail.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Facetime-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Facetime-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Facetime.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Hangouts-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Hangouts-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Hangouts.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Map-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Map-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Map.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Skype-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Skype-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Skype.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_SMS-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_SMS-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_SMS.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Tel-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Tel-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/PhoneActionType_Tel.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Checked.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Download-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Download-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Navi-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Navi-Normal.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Uncheck.png"
  install_resource "${PODS_ROOT}/WCCardHolderCell/Assets/WCCardHolderCell_Unverified.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/NoteViewControllerButtonDate-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/NoteViewControllerButtonDate-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactControllerButtonGroupUnSynced.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactMoreButtonDelete-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactMoreButtonDelete-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactMoreButtonNote-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactMoreButtonNote-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonChangeSide-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonChangeSide-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonHideKeyboard-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonMore-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonMore-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonSave-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/WCContactViewControllerButtonSave-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditDefaultHead.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditMoreButtonChange-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditMoreButtonChange-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditMoreButtonClipAdd-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditMoreButtonClipAdd-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddFrontCard-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddFrontCard-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddNext-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddNext-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddRearCard-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonAddRearCard-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonFlip-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonFlip-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonNote-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactEdit/WCContactEditViewControllerButtonNote-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoDefaultHead-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonCommunitySearch-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonCommunitySearch-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonEdit-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonEdit-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonRecog-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonRecog-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonRefreshSocialNetwork-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonRefreshSocialNetwork-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonSaveContact-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonSaveContact-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonSendContact-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonSendContact-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonShareContact-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonShareContact-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonTranslate-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoMoreButtonTranslate-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonFavorite-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonFavorite-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonFavorite-Selected.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonLanguageArrow-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonLanguageArrow-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonMoreLanguage-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonMoreLanguage-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonNote-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerButtonNote-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellAddress-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellBirthday-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellCompany-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellContactName-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellDate-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellEmail-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellGroup-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellIM-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellOthers-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellPhone-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellSMS-Highlighted.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellSMS-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellSocialNetwork-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellURL-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellUserDefine-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Image/ContactInfo/WCContactInfoViewControllerCellViewAccount-Normal.png"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCContactViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarAddress-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarAddress-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarAddress-Selected.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarGo-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarGo-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarMail-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarMail-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarMail-Selected.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarNext-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarNext-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarPrev-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarPrev-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarTel-Highlighted.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarTel-Normal.png"
  install_resource "${PODS_ROOT}/WCCoverFlowViewController/Assets/WCCoverFlowBarTel-Selected.png"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCDataModel/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Image/WCExportFlowButtonExport-Highlighted.png"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Image/WCExportFlowButtonExport-Normal.png"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCExportFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/ic_check-box_select_h.png"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/ic_check-box_unselect_h.png"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCFileTypeSelectView/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupAdd-Highlighted.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupAdd-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupDone-Highlighted.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupDone-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupSort-Highlighted.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/GroupSort-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/de.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/en.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/es.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/fr.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/it.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/ja.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/ko.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/nl.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/pt.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCGroupAccessFlowController/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/de.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/en.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/es.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/fr.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/it.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/ja.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/ko.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/nl.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/pt.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCGroupAccessFlowController/Assets/WCSingleLayerGroupViewController/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/Collapse-Highlighted.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/Collapse-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/Expand-Highlighted.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/Expand-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/Pin-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/UnPin-Normal.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/WCGroupTableViewCell_Checked.png"
  install_resource "${PODS_ROOT}/WCGroupView/Assets/Icons/WCGroupTableViewCell_Uncheck.png"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Image/GroupViewControllerButtonImport-Highlighted.png"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Image/GroupViewControllerButtonImport-Normal.png"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCImportFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerButtonLocation-Highlighted.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerButtonLocation-Normal.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerButtonOther-Normal.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerButtonSelf-Normal.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerDefaultHead.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerTableViewCellDisclosureIndicator-Highlighted.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Image/WCMapViewControllerTableViewCellDisclosureIndicator-Normal.png"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCMapViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/PPSignatureConrollerButtonRecog-Highlighted.png"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/PPSignatureConrollerButtonRecog-Normal.png"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/PPSignatureConrollerButtonTip-Highlighted.png"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/PPSignatureConrollerButtonTip-Normal.png"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/de.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/en.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/es.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/fr.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/it.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/ja.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/ko.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/nl.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/pt.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCNewCardFlowController/Assets/PPSignatureViewController/String/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCRecognitionController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCSearchViewController/Assets/Images/WCSearchViewControllerButton_AdvanceSearch-Normal.png"
  install_resource "${PODS_ROOT}/WCSearchViewController/Assets/Images/WCSearchViewControllerButton_SearchClear-Highlighted@3x.png"
  install_resource "${PODS_ROOT}/WCSearchViewController/Assets/Images/WCSearchViewControllerButton_SearchClear-Normal@3x.png"
  install_resource "${PODS_ROOT}/WCSearchViewController/Assets/Images/WCSearchViewControllerButton_SearchIcon@3x.png"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Image/WCShareFlowButtonShare-Highlighted.png"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Image/WCShareFlowButtonShare-Normal.png"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCShareFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTAccountDataController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTAccountFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoMoreButtonEdit-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoMoreButtonEdit-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoMoreButtonTemplate-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoMoreButtonTemplate-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoShareCellEmail-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoShareCellQRCode-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Image/AccountInfoShareCellSMS-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTAccountInfoViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Image/AccountSelectDone-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Image/AccountSelectDone-Normal.png"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Image/ic_check-box_select_h.png"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Image/ic_check-box_unselect_h.png"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTAccountSelectViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTAdvancedSearchViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/VisibleAccount-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/VisibleAccount-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonChangeOwner-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonChangeOwner-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDelete-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDelete-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownload-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownload-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloaded-Disabled.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloaded-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloaded-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloading-Disabled.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloading-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonDownloading-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonNeedDownload-Disabled.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonNeedDownload-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonNeedDownload-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonShareToColleague-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCCompanyContactViewControllerButtonShareToColleague-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCTCompanyMultiSelect-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCTCompanyMultiSelect-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCTCompanySearch-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Images/WCTCompanySearch-Normal.png"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTCompanyContactViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTLoginController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonDrop-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonDrop-Normal.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonHidePassword-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonHidePassword-Normal.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonShowPassword-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerButtonShowPassword-Normal.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerContentBackground.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Image/WCTLoginViewControllerHeaderBackground.png"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTLoginViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTNoticePageViewController_Filter-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTNoticePageViewController_Filter-Normal.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTSharedContactNoticeViewController_Download-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTSharedContactNoticeViewController_Download-Normal.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTSharedContactNoticeViewController_SelectAll-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Icon/WCTSharedContactNoticeViewController_SelectAll-Normal.png"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTNoticeFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Image/WCTPasswordViewControllerButtonDone-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Image/WCTPasswordViewControllerButtonDone-Normal.png"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTPasswordViewController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTRestClientController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Images/WCTELVC_ButtonHidePassword-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Images/WCTELVC_ButtonHidePassword-Normal.png"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Images/WCTELVC_ButtonShowPassword-Highlighted.png"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Images/WCTELVC_ButtonShowPassword-Normal.png"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Classes/Private/WCTQNapLoginViewController.xib"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Classes/Private/WCTQNapQuestionViewController.xib"
  install_resource "${PODS_ROOT}/WCTServerContactExportFlowController/Classes/Private/WCTQNapSecurityCodeViewController.xib"
  install_resource "${PODS_ROOT}/WCXFDataController/Assets/wcxficon.png"
  install_resource "${PODS_ROOT}/WCXFDataController/Assets/wcxficon@2x.png"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/de.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/en.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/es.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/fr.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/it.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/ja.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/ko.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/nl.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/pt.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/zh-Hans.lproj"
  install_resource "${PODS_ROOT}/WCXLSXDataController/Assets/Localizable/zh-Hant.lproj"
  install_resource "${PODS_ROOT}/libIndexing/Assets/Default/ppkkc.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/amedict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/arabicdict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/bphrase1.big"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/bphrase1.gbk"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/bphrase1.jap"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/bphrase1.ksc"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/cocr_dts"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/cocr_sim"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/cocr_trad"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_ar.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_br.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_eeu.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_eng.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_gr.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_neu.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_proto_th.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_rus.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_th.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_tky.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/dict_weu.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/engdict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/esteupdict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/eupdict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/gb-big5.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/grdict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/jocr"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/jocra"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/kocr"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/neudict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phb5.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phceng.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phgb.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phjeng.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phjp.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/phks.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/russiadict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/surname.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/thaidict.dat"
  install_resource "${PODS_ROOT}/libbcrsdk/Assets/iOS/database/turkeydict.dat"
  install_resource "${PODS_CONFIGURATION_BUILD_DIR}/nanopb/nanopb_Privacy.bundle"
fi

mkdir -p "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"
rsync -avr --copy-links --no-relative --exclude '*/.svn/*' --files-from="$RESOURCES_TO_COPY" / "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"
if [[ "${ACTION}" == "install" ]] && [[ "${SKIP_INSTALL}" == "NO" ]]; then
  mkdir -p "${INSTALL_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"
  rsync -avr --copy-links --no-relative --exclude '*/.svn/*' --files-from="$RESOURCES_TO_COPY" / "${INSTALL_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"
fi
rm -f "$RESOURCES_TO_COPY"

if [[ -n "${WRAPPER_EXTENSION}" ]] && [ "`xcrun --find actool`" ] && [ -n "${XCASSET_FILES:-}" ]
then
  # Find all other xcassets (this unfortunately includes those of path pods and other targets).
  OTHER_XCASSETS=$(find -L "$PWD" -iname "*.xcassets" -type d)
  while read line; do
    if [[ $line != "${PODS_ROOT}*" ]]; then
      XCASSET_FILES+=("$line")
    fi
  done <<<"$OTHER_XCASSETS"

  if [ -z ${ASSETCATALOG_COMPILER_APPICON_NAME+x} ]; then
    printf "%s\0" "${XCASSET_FILES[@]}" | xargs -0 xcrun actool --output-format human-readable-text --notices --warnings --platform "${PLATFORM_NAME}" --minimum-deployment-target "${!DEPLOYMENT_TARGET_SETTING_NAME}" ${TARGET_DEVICE_ARGS} --compress-pngs --compile "${BUILT_PRODUCTS_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"
  else
    printf "%s\0" "${XCASSET_FILES[@]}" | xargs -0 xcrun actool --output-format human-readable-text --notices --warnings --platform "${PLATFORM_NAME}" --minimum-deployment-target "${!DEPLOYMENT_TARGET_SETTING_NAME}" ${TARGET_DEVICE_ARGS} --compress-pngs --compile "${BUILT_PRODUCTS_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}" --app-icon "${ASSETCATALOG_COMPILER_APPICON_NAME}" --output-partial-info-plist "${TARGET_TEMP_DIR}/assetcatalog_generated_info_cocoapods.plist"
  fi
fi
